//
//  SDKAutoUpgrade.cpp
//  AriesX
//
//  Created by mingfyan on 15/4/24.
//  Copyright (c) 2015 Cisco Systems. All rights reserved.
//

#include "SDKAutoUpgrade.h"

#include "csf/http/BasicHttpClient.hpp"
#include "csf/http/HttpClientListener.hpp"
#include "csf/http/Response.hpp"
#include "csf/netutils/NetUtilsFactory.hpp"
#include "csf/http/EdgePolicy.hpp"
#include "csf/http/HttpRequestOptions.hpp"
#include "csf/Mutex.hpp"
#include "csf/ScopedLock.hpp"

#include "csfunified/services/interface/ConfigService.h"

#include "jcfcoreutils/FileUtils.h"
#include "jcfcoreutils/StringUtils.h"
#include "jcfcoreutils/SystemUtils.h"
#include "jcfcoreutils/ThreadUtils.h"
#include "jcfcoreutils/TimeUtils.h"

#include "boost/algorithm/string.hpp"
#include "csf/logger/CSFLogger.hpp"

#include "openssl/crypto.h"
#include "openssl/sha.h"
#include "openssl/pem.h"

#include "../services/impl/MeetingTelemetry.h"
#include "MeetingSDKMgr.h"

#include <sstream>
#include <iomanip>
#include <vector>

#ifdef _WIN32
#include <direct.h>
#include <io.h>
#else
#include <unistd.h>
#include <utime.h>
#endif

// header files for PE signature check
#ifdef _WIN32
#define _UNICODE 1
#define UNICODE 1

#include <windows.h>
#include <Softpub.h>
#include <wincrypt.h>
#include <wintrust.h>

// Link with the Wintrust.lib file.
#pragma comment (lib, "wintrust")
#else
#include <stdlib.h> /* system method */
#endif

// header file for chmod
#include <sys/stat.h>
#ifdef _WIN32
typedef int mode_t;
#endif


using namespace JCFCoreUtils;
using namespace csf::http;

extern "C"
{
#include "jansson.h"
}

const std::string SDKUpdateQuery::SUCCESS_KEY = "success";
const std::string SDKUpdateQuery::UPGRADE_KEY = "upgrade";
const std::string SDKUpdateQuery::VERSION_KEY = "version";
const std::string SDKUpdateQuery::FILEURL_KEY = "fileUrl";
const std::string SDKUpdateQuery::HASHKEY_KEY = "hashKey";

const std::string LocalSDKConfigs::CHECKTIME_KEY = "LastUpgradeCheckTime";
const std::string LocalSDKConfigs::TGT_VER_KEY = "TargetVersion";
const std::string LocalSDKConfigs::TGT_LOCATION_KEY = "TargetLocation";


static CSFLogger* CSDKAutoUpgradeLogger = CSFLogger_getLogger("MeetingService-CSDKAutoUpgrade");

static csf::Mutex mutex;

const char MeetingSDKVersion::delim = '.';
std::string MeetingSDKVersion::ToString()
{
    std::stringstream ss;
    ss << major << delim << middle << delim << minor;
    return ss.str();
}

void MeetingSDKVersion::FromString(std::string versionStr)
{
    std::istringstream iss(versionStr);
    std::string token;
    
    std::getline(iss, token, delim);
    if(token.empty())
    {
        return;
    }
    major = toInt(token, 0);
    
    std::getline(iss, token, delim);
    if(token.empty())
    {
        return;
    }
    middle = toInt(token, 0);
    
    std::getline(iss, token, delim);
    if(token.empty())
    {
        return;
    }
    minor = toInt(token, 0);
}

bool MeetingSDKVersion::isLargerThan(const MeetingSDKVersion& version)
{
    if (major > version.major)
    {
        return true;
    }
    else if(major < version.major)
    {
        return false;
    }
    
    if (middle > version.middle)
    {
        return true;
    }
    else if(middle < version.middle)
    {
        return false;
    }
    
    if (minor > version.minor)
    {
        return true;
    }
    else if(minor < version.minor)
    {
        return false;
    }
    
    if (build > version.build)
    {
        return true;
    }
    else if(build < version.build)
    {
        return false;
    }
    
    return false;
}

bool MeetingSDKVersion::isEqualTo(const MeetingSDKVersion& version)
{
    return major == version.major && middle == version.middle && minor == version.minor && build == version.build;
}

bool SDKUpdateQuery::ParseQueryReponse(std::string text, SDKUpdateQuery& result)
{
    json_error_t error;
    json_t *root = json_loads(text.c_str(), 0, &error);
    if (!root)
    {
        CSFLogErrorS(CSDKAutoUpgradeLogger, "Failed to load text as Json: " << text);
        return false;
    }
    
    const char* successStr;
    const char* upgradeStr;
    std::string tmpStr;
    
    int ret;
    ret= json_unpack(root, "{s:s, s:s}", SUCCESS_KEY.c_str(), &successStr, UPGRADE_KEY.c_str(), &upgradeStr);
    if (0 == ret)
    {
        tmpStr = successStr;
        result.isSuccess = boost::iequals(tmpStr, "true");
        tmpStr = upgradeStr;
        result.isUpgrade = boost::iequals(tmpStr, "true");
        
        if (!result.isUpgrade)
        {
            json_decref(root);
            return true;
        }
    }
    else
    {
        json_decref(root);
        return false;
    }
    
    const char* versionStr;
    const char* urlStr;
    const char* hashStr;
    ret = json_unpack(root, "{s:s, s:s, s:s, s:s, s:s}", SUCCESS_KEY.c_str(), &successStr, UPGRADE_KEY.c_str(), &upgradeStr, VERSION_KEY.c_str(), &versionStr, FILEURL_KEY.c_str(), &urlStr, HASHKEY_KEY.c_str(), &hashStr);
    if (0 != ret)
    {
        CSFLogErrorS(CSDKAutoUpgradeLogger, "Failed to parse text as expected format: " << text);
        json_decref(root);
        return false;
    }
    
    tmpStr = successStr;
    result.isSuccess = boost::iequals(tmpStr, "true");
    tmpStr = upgradeStr;
    result.isUpgrade = boost::iequals(tmpStr, "true");
    
    result.version = versionStr;
    result.fileUrl = urlStr;
    result.hashKey = hashStr;
    
    json_decref(root);
    
    if (result.version.empty() || result.fileUrl.empty() || result.hashKey.empty())
    {
        CSFLogErrorS(CSDKAutoUpgradeLogger, "Downloaded meeting SDK properties should not be empty: " << text);
        return false;
    }
    
    return true;
}

LocalSDKConfigs* LocalSDKConfigs::m_pInstance = NULL;
bool LocalSDKConfigs::loadFromFile(std::string configFile)
{
    if (configFile.empty())
    {
        return false;
    }
    
    json_error_t error;
    json_t *root = json_load_file(configFile.c_str(), 0, &error);
    if (!root)
    {
        CSFLogErrorS(CSDKAutoUpgradeLogger, "Failed to load SDK config file: " << configFile);
        return false;
    }
    
    const char* versionStr;
    const char* locationStr;
    int ret = json_unpack(root, "{s:I, s:s, s:s}", CHECKTIME_KEY.c_str(), &lastCheckTime, TGT_VER_KEY.c_str(), &versionStr, TGT_LOCATION_KEY.c_str(), &locationStr);
    
    if (0 != ret)
    {
        CSFLogErrorS(CSDKAutoUpgradeLogger, "Failed to parse SDK config file: " << configFile);
        json_decref(root);
        return false;
    }
    
    targetVer = versionStr;
    targetLocation = locationStr;
    
    json_decref(root);
    
    return true;
}

bool LocalSDKConfigs::dumpToFile(std::string configFile)
{
    if (configFile.empty())
    {
        return false;
    }
    
    std::string parentFolder = FileUtils::getParentDirectory(configFile);
    if (!FileUtils::isDirectory(parentFolder))
    {
        FileUtils::createDirectory(parentFolder);
    }
    
    json_t* jsonContent = json_pack("{s:I, s:s, s:s}", CHECKTIME_KEY.c_str(), lastCheckTime, TGT_VER_KEY.c_str(), targetVer.c_str(), TGT_LOCATION_KEY.c_str(), targetLocation.c_str());
    
    return json_dump_file(jsonContent, configFile.c_str(), 0) == 0;
}

TaskDelayTrigger::TaskDelayTrigger(long delaySeconds, csf::TimerCallback* task, long executeInterval)
{
    m_task = task;
    m_interval = executeInterval;
    
    m_triggerTimer = NULL;
    m_taskTimer = NULL;
    
    int result = 0;
    
    m_triggerTimer = new csf::TimerHelper(this, delaySeconds * 1000, false, result);
    if (!m_triggerTimer || result != 0)
    {
        CSFLogWarnS(CSDKAutoUpgradeLogger, "Failed to start delay trigger");
    }
}

void TaskDelayTrigger::onTimerFired(csf::TimerHelper *timer)
{
    int result = 0;
    
    m_taskTimer = new csf::TimerHelper(m_task, m_interval * 1000, true, result);
    if (!m_triggerTimer || result != 0)
    {
        CSFLogWarnS(CSDKAutoUpgradeLogger, "Failed to start delay task");
    }
}

TaskDelayTrigger::~TaskDelayTrigger()
{
    if (m_taskTimer)
    {
        delete m_taskTimer;
    }
    
    if(m_triggerTimer)
    {
        delete m_triggerTimer;
    }
}

AutoUpgradeTask::AutoUpgradeTask(IAutoUpgradeHandler* pAutoUpgradeHandler)
{
    m_pAutoUpgradeHandler = pAutoUpgradeHandler;
}

void AutoUpgradeTask::onTimerFired(csf::TimerHelper *timer)
{
    // check whether the time from last check is larger than defined interval
    long long currentTimeInSeconds = TimeUtils::getCurrentTimeInSeconds();
    long long interval = currentTimeInSeconds - m_pAutoUpgradeHandler->getLastCheckTime();
    
    if (interval >= m_pAutoUpgradeHandler->getUpgradeCheckInterval())
    {
        ThreadUtils::runInThreadPool( std::bind(&AutoUpgradeTask::ScanAndDownload, this) , "Scan and download new Meeting SDK package task" );
    }
}

void AutoUpgradeTask::ScanAndDownload()
{
    csf::ScopedLock lock(mutex);
    
    if (!m_pAutoUpgradeHandler) {
        CSFLogWarnS(CSDKAutoUpgradeLogger, "Failed to start scan and download, since the upgrade handler is not initialized.");
        return;
    }
    
    bool isNeedDownload = false;

	MeetingSDKVersion localSDKVersion;
	std::string userDomain;
	MeetingSDKMgr::getInstance()->getLatestSDKVersion(localSDKVersion);
	userDomain = MeetingSDKMgr::getInstance()->getUserDomain();
    
	CustomizedSyncHttpQuery httpQuery(m_pAutoUpgradeHandler->getNetUtilTransportDownloader());
    SDKUpdateQuery queryResult;
    queryResult.currentTimeInSeconds = TimeUtils::getCurrentTimeInSeconds();
    do
    {
        std::string monitorUrl = m_pAutoUpgradeHandler->getMonitorUrl();
        std::string retContent;
		bool bResult = httpQuery.HttpQuery(monitorUrl, retContent);
        if (!bResult) {
            CSFLogWarnS(CSDKAutoUpgradeLogger, "Failed to connect for query: " << monitorUrl);
            queryResult.isSuccess = false;

			CMeetingTelemetry telemetry;
			telemetry.addMember("host_domain", userDomain);
			telemetry.addMember("current_version", localSDKVersion.ToString());
			telemetry.addMember("upgrade_result", "failure");
			telemetry.addMember("failure_reason", "network block");
			telemetry.addMember("failure_phase", "check meetingsdk phase");
			telemetry.upload("meetingupgrade");

            break;
        }
        
        bResult = SDKUpdateQuery::ParseQueryReponse(retContent, queryResult);
        if (!bResult) {
            queryResult.isSuccess = false;
            CSFLogWarnS(CSDKAutoUpgradeLogger, "Failed to parse SDK query return string: " << retContent);

			CMeetingTelemetry telemetry;
			telemetry.addMember("host_domain", userDomain);
			telemetry.addMember("current_version", localSDKVersion.ToString());
			telemetry.addMember("upgrade_result", "failure");
			telemetry.addMember("failure_reason", "response has error");
			telemetry.addMember("failure_phase", "check meetingsdk phase");
			telemetry.upload("meetingupgrade");

            break;
        }
        
        if (!queryResult.isSuccess || !queryResult.isUpgrade) {
            CSFLogWarnS(CSDKAutoUpgradeLogger, "No need to upgrade --  isSuccess: " << queryResult.isSuccess << "  isUpgrade: " << queryResult.isUpgrade);

			CMeetingTelemetry telemetry;
			telemetry.addMember("host_domain", userDomain);
			telemetry.addMember("current_version", localSDKVersion.ToString());
			telemetry.addMember("upgrade_version", queryResult.version);
			telemetry.addMember("need_upgrade", "no");
			telemetry.upload("meetingupgrade");

			break;
        }
        
        isNeedDownload = true;
        
    }while (0);
    m_pAutoUpgradeHandler->onQuery(queryResult);
    
    if (!isNeedDownload) {
        return;
    }
    
    std::string archiveFile = m_pAutoUpgradeHandler->getDownloadZipPath();
    if (FileUtils::fileExists(archiveFile))
    {
        FileUtils::removeFile(archiveFile);
    }
    std::string parentFolder = FileUtils::getParentDirectory(archiveFile);
    if (!FileUtils::isDirectory(parentFolder))
    {
        FileUtils::createDirectory(parentFolder);
    }
    
    csf::Event* pCompleteEvent = m_pAutoUpgradeHandler->getDownloadCompleteEvent();
    
    pCompleteEvent->reset();

    m_pAutoUpgradeHandler->getNetUtilTransportDownloader()->DownloadToFile(queryResult.fileUrl, archiveFile,
		std::bind(&IAutoUpgradeHandler::downloadToFileComplete, m_pAutoUpgradeHandler, std::placeholders::_1), 0);
    
    pCompleteEvent->timedWait(m_pAutoUpgradeHandler->getDownloadWaitTimeoutSeconds() * 1000);
}

void UnzipUtils::change_file_date(const char * filename, uLong dosdate, tm_unz tmu_date)
{
#ifdef _WIN32
    HANDLE hFile;
    FILETIME ftm,ftLocal,ftCreate,ftLastAcc,ftLastWrite;
    
    hFile = CreateFileA(filename,GENERIC_READ | GENERIC_WRITE,
                       0,NULL,OPEN_EXISTING,0,NULL);
    GetFileTime(hFile,&ftCreate,&ftLastAcc,&ftLastWrite);
    DosDateTimeToFileTime((WORD)(dosdate>>16),(WORD)dosdate,&ftLocal);
    LocalFileTimeToFileTime(&ftLocal,&ftm);
    SetFileTime(hFile,&ftm,&ftLastAcc,&ftm);
    CloseHandle(hFile);
#else
    struct utimbuf ut;
    struct tm newdate;
    newdate.tm_sec = tmu_date.tm_sec;
    newdate.tm_min=tmu_date.tm_min;
    newdate.tm_hour=tmu_date.tm_hour;
    newdate.tm_mday=tmu_date.tm_mday;
    newdate.tm_mon=tmu_date.tm_mon;
    if (tmu_date.tm_year > 1900)
        newdate.tm_year=tmu_date.tm_year - 1900;
    else
        newdate.tm_year=tmu_date.tm_year ;
    newdate.tm_isdst=-1;
    
    ut.actime=ut.modtime=mktime(&newdate);
    utime(filename,&ut);
#endif
}

bool UnzipUtils::UnZipWithoutPassword(std::string zipPath, std::string extractFolderPath)
{
    return UnZipWithPassword(zipPath, extractFolderPath, "");
}

bool UnzipUtils::UnZipWithPassword(std::string zipPath, std::string extractFolderPath, std::string password)
{
    unzFile _unzFile = unzOpen(zipPath.c_str());
	if (!_unzFile)
	{
		return false;
	}
    
	bool success = true;
    int ret = unzGoToFirstFile(_unzFile);
    if( ret != UNZ_OK )
    {
		if (_unzFile)
		{
			unzClose(_unzFile);
		}
        return false;
    }
    
    do{
        if (password.empty())
        {
            ret = unzOpenCurrentFile( _unzFile );
        }
        else
        {
            ret = unzOpenCurrentFilePassword( _unzFile, password.c_str());
        }
        
        if( ret != UNZ_OK )
        {
            success = false;
			break;
        }

        // reading data and write to file
        unz_file_info fileInfo = {0};
        ret = unzGetCurrentFileInfo(_unzFile, &fileInfo, NULL, 0, NULL, 0, NULL, 0);
        if( ret != UNZ_OK )
        {
			success = false;
            unzCloseCurrentFile( _unzFile );
            break;
        }
        
        char* filename = (char*) malloc( fileInfo.size_filename + 1 );
        unzGetCurrentFileInfo(_unzFile, &fileInfo, filename, fileInfo.size_filename + 1, NULL, 0, NULL, 0);
        filename[fileInfo.size_filename] = '\0';
        
        std::string strPath = filename;
        
        // check if it contains directory
        bool isDirectory = filename[fileInfo.size_filename-1] == '/' || filename[fileInfo.size_filename-1] == '\\';
        free( filename );
        
        std::string fullPath = extractFolderPath + FileUtils::FileSeparator() + strPath;
        if( isDirectory )
        {
            FileUtils::createDirectory(fullPath);
        }
        else
        {
            FileUtils::createParentDirectory(fullPath);
        }
        
        FILE* fp = fopen(fullPath.c_str(), "wb");
        const int buff_size = 4096;
        unsigned char buffer[buff_size] = {0};
        while( fp )
        {
            int read = unzReadCurrentFile(_unzFile, buffer, buff_size);
            if( read > 0 )
            {
                fwrite(buffer, read, 1, fp );
            }
            else if( read<0 )
            {
                break;
            }
            else
                break;
        }
        if( fp )
        {
            fclose( fp );
            
            // restore permission
            mode_t filemode = 0xFFFF & (fileInfo.external_fa >> 16L);
#ifdef WIN32
            _chmod(fullPath.c_str(), filemode);
#else
            chmod(fullPath.c_str(), filemode);
#endif
            
            // set the orignal datetime property
            if( fileInfo.dosDate != 0 )
            {
                change_file_date(fullPath.c_str(), fileInfo.dosDate, fileInfo.tmu_date);
            }
            
        }
        unzCloseCurrentFile( _unzFile );
        ret = unzGoToNextFile( _unzFile );
        
    }while( ret==UNZ_OK && UNZ_OK!=UNZ_END_OF_LIST_OF_FILE );

	if (_unzFile)
	{
		return unzClose(_unzFile) == UNZ_OK;
	}

    return success;
}

std::string FileCheckerUtils::calc_file_sha256(const std::string& filePath)
{
    FILE* file = fopen(filePath.c_str(), "rb");
    if(!file)
    {
        return "";
    }
    
    unsigned  char hash[SHA256_DIGEST_LENGTH];
    SHA256_CTX sha256;
    SHA256_Init(&sha256);
    const int bufSize = 32768;
    char* buffer = new char[bufSize];
    int bytesRead = 0;
    if(!buffer)
    {
        fclose(file);
        delete [] buffer;
        return "";
    }
    while((bytesRead = fread(buffer, 1, bufSize, file)))
    {
        SHA256_Update(&sha256, buffer, bytesRead);
    }
    SHA256_Final(hash, &sha256);
    
    std::stringstream ss;
    for(int i = 0; i < SHA256_DIGEST_LENGTH; i++)
    {
        ss << std::hex << std::setw(2) << std::setfill('0') << (int)hash[i];
    }
    
    fclose(file);
    delete [] buffer;
    
    return ss.str();
}

bool FileCheckerUtils::checkSignatureForFolderFiles(const std::string& folder)
{
    std::vector<std::string> sigFileExtensions;
#ifdef _WIN32
    sigFileExtensions.push_back(".dll");
    sigFileExtensions.push_back(".exe");
    sigFileExtensions.push_back(".DLL");
    sigFileExtensions.push_back(".EXE");
#else
    sigFileExtensions.push_back(".dylib");
    sigFileExtensions.push_back(".DYLIB");
#endif
    
    for (std::vector<std::string>::iterator iter = sigFileExtensions.begin(); iter != sigFileExtensions.end(); iter++)
    {
        std::vector<std::string> files;
        FileUtils::getFilesFromDirectory(folder, *iter, files);
        
        for (std::vector<std::string>::iterator fileIter = files.begin(); fileIter != files.end(); fileIter++)
        {
            if(!checkSignature(*fileIter))
            {
                return false;
            }
        }
    }
    
    return true;
}


bool FileCheckerUtils::checkSignature(const std::string& filePath)
{
#ifdef _WIN32
    // Reference: https://msdn.microsoft.com/en-us/library/aa382384%28VS.85%29.aspx
    
    std::wstring file = toWideString(filePath);
    
    LONG lStatus;
    
    // Initialize the WINTRUST_FILE_INFO structure.
    WINTRUST_FILE_INFO FileData;
    memset(&FileData, 0, sizeof(FileData));
    FileData.cbStruct = sizeof(WINTRUST_FILE_INFO);
    FileData.pcwszFilePath = file.c_str();
    FileData.hFile = NULL;
    FileData.pgKnownSubject = NULL;
    
    // Initialize the WinVerifyTrust input data structure.
    WINTRUST_DATA WinTrustData;
    memset(&WinTrustData, 0, sizeof(WinTrustData));
    WinTrustData.cbStruct = sizeof(WinTrustData);
    WinTrustData.pPolicyCallbackData = NULL;
    WinTrustData.pSIPClientData = NULL;
    WinTrustData.dwUIChoice = WTD_UI_NONE;
    WinTrustData.fdwRevocationChecks = WTD_REVOKE_NONE;
    WinTrustData.dwUnionChoice = WTD_CHOICE_FILE;
    WinTrustData.dwStateAction = WTD_STATEACTION_VERIFY;
    WinTrustData.hWVTStateData = NULL;
    WinTrustData.pwszURLReference = NULL;
    WinTrustData.dwUIContext = 0;
    
    // Set pFile.
    WinTrustData.pFile = &FileData;
    
    // WinVerifyTrust verifies signatures as specified by the GUID
    // and Wintrust_Data.
    GUID WVTPolicyGUID = WINTRUST_ACTION_GENERIC_VERIFY_V2;
    lStatus = WinVerifyTrust(
                             NULL,
                             &WVTPolicyGUID,
                             &WinTrustData);
    
    bool success = lStatus == ERROR_SUCCESS;
    
    // Any hWVTStateData must be released by a call with close.
    WinTrustData.dwStateAction = WTD_STATEACTION_CLOSE;
    
    lStatus = WinVerifyTrust(
                             NULL,
                             &WVTPolicyGUID,
                             &WinTrustData);
    
    return success;
#else
    // 2015.5.7 Mingfeng: haven't found pure C++ code to verify PE codesign on Mac. There is Objective C++ code which uses system library. Not apply it yet.
    
    std::string cmd = "codesign --verify ";
    cmd += "\"" + filePath + "\"";
    
    int ret = system(cmd.c_str());
    return ret == 0;
#endif
}


